﻿// The MIT License (MIT)
// Copyright (c) 2014 Nicolas Kogler

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.

using System.Drawing;
using System.Drawing.Text;
using System.Windows.Forms;
using System.ComponentModel;
using System.Drawing.Imaging;
using System.Drawing.Drawing2D;
 

namespace IMLibrary.Controls.Win8form
{
    public class SharpToolRenderer : ToolStripSystemRenderer
    {
        /// <summary>
        /// With this, we draw the background of the toolstrip.
        /// </summary>
        public Color FormBackColor { get; set; }

        /// <summary>
        /// Draws the background of the DropDownMenu.
        /// </summary>
        protected override void OnRenderImageMargin(ToolStripRenderEventArgs e)
        {
            var bounds = e.AffectedBounds;
            var height = e.ToolStrip.Height;
            var width = e.ToolStrip.Width;

            var borderrect = new Rectangle(0, 0, width - 1, height - 1);
            var backrect = new Rectangle(1, 1, width - 2, height - 2);
            var point2 = new Point(bounds.Width, bounds.Height);
            var point1 = new Point(bounds.Width, 0);

            var backbrush = new SolidBrush(Color.FromArgb(255, 254, 254, 254));
            var borderpen = new Pen(Color.FromArgb(255, 77, 77, 77));

            e.Graphics.FillRectangle(backbrush, backrect);
            e.Graphics.FillRectangle(backbrush, bounds);
            e.Graphics.DrawRectangle(borderpen, borderrect);
            e.Graphics.DrawLine(borderpen, point1, point2);

            backbrush.Dispose();
            borderpen.Dispose();
        }

        /// <summary>
        /// We don't need border, if we draw the toolstrip
        /// on the form-border background -> override it.
        /// </summary>
        protected override void OnRenderToolStripBorder(ToolStripRenderEventArgs e)
        {
            if (e.ToolStrip.IsDropDown)
            {
                base.OnRenderToolStripBorder(e);
            }
        }

        /// <summary>
        /// We also don't want a background drawn on
        /// the form-border background, so we clear
        /// the whole area with a transparent color.
        /// </summary>
        protected override void OnRenderToolStripBackground(ToolStripRenderEventArgs e)
        {
            if (e.ToolStrip.IsDropDown)
            {
                base.OnRenderToolStripBackground(e);
            }
            else
            {
                e.Graphics.Clear(FormBackColor);
            }
        }

        /// <summary>
        /// When having the aero-glass style enabled, text
        /// or images get pretty much messed up when anti-aliased.
        /// That's why we use a GraphicsPath to draw it smoothly.
        /// </summary>
        /// <param name="e"></param>
        protected override void OnRenderItemText(ToolStripItemTextRenderEventArgs e)
        {
            if (e.ToolStrip.IsDropDown)
            {
                base.OnRenderItemText(e);
            }
            else
            {
                using (var path = new GraphicsPath())
                {
                    path.AddString(e.Text, e.TextFont.FontFamily, (int)e.TextFont.Style,
                        e.TextFont.Size + 2, e.TextRectangle, StringFormat.GenericDefault);

                    e.Graphics.SmoothingMode = SmoothingMode.HighQuality;
                    e.Graphics.FillPath(SystemBrushes.WindowText, path);
                }
            }
        }

        /// <summary>
        /// The system-drawn separator looks ugly in my
        /// opinion, so we're gonna redraw it in this method.
        /// </summary>
        protected override void OnRenderSeparator(ToolStripSeparatorRenderEventArgs e)
        {
            if (e.Vertical)
            {
                var linebrush1 = new SolidBrush(Color.FromArgb(255, 180, 180, 180));
                var linebrush2 = new SolidBrush(Color.FromArgb(150, 254, 254, 254));
                var rectangle1 = new Rectangle(2, 4, 1, e.Item.Height - 8);
                var rectangle2 = new Rectangle(3, 4, 1, e.Item.Height - 8);

                e.Graphics.FillRectangle(linebrush1, rectangle1);
                e.Graphics.FillRectangle(linebrush2, rectangle2);

                linebrush1.Dispose();
                linebrush2.Dispose();
            }
            else
            {
                var linerect = new Rectangle(24, 3, e.Item.Width, 1);
                var linebrush = new SolidBrush(Color.FromArgb(255, 77, 77, 77));

                e.Graphics.FillRectangle(linebrush, linerect);
                linebrush.Dispose();
            }
        }

        protected override void OnRenderButtonBackground(ToolStripItemRenderEventArgs e)
        {
            this.OnRenderItemBackground(e);
        }

        protected override void OnRenderItemBackground(ToolStripItemRenderEventArgs e)
        {
            if (!e.Item.IsOnDropDown)
            {
                if (e.Item.Selected)
                {
                    var backbrush = new SolidBrush(((SharpToolStrip)e.ToolStrip).SelectedBackColor);
                    var borderpen = new Pen(((SharpToolStrip)e.ToolStrip).SelectedBorderColor);
                    var rectangle = new Rectangle(0, 0, e.Item.Width - 1, e.Item.Height - 1);

                    e.Graphics.FillRectangle(backbrush, rectangle);
                    e.Graphics.DrawRectangle(borderpen, rectangle);

                    backbrush.Dispose();
                    borderpen.Dispose();
                }
                if (e.Item.Pressed)
                {
                    var backbrush = new SolidBrush(((SharpToolStrip)e.ToolStrip).ClickedBackColor);
                    var borderpen = new Pen(((SharpToolStrip)e.ToolStrip).ClickedBorderColor);
                    var rectangle = new Rectangle(0, 0, e.Item.Width - 1, e.Item.Height - 1);

                    e.Graphics.FillRectangle(backbrush, rectangle);
                    e.Graphics.DrawRectangle(borderpen, rectangle);

                    backbrush.Dispose();
                    borderpen.Dispose();
                }
            }
            else
            {
                base.OnRenderItemBackground(e);
            }
        }
    }
}